################################################################################
# R script for two-class LDA using the SHIP covariance estimator. (Note that   #
# the pooled centroids formulation of multiclass LDA from Ahdesmki and        #
# Strimmer (2010) is employed.)                                                #
################################################################################

######################
# Function rldaCMA() #
######################

rldaCMA   <- function(X,y,learnind,type,genesINpaths=NULL,models=FALSE) {
  nrowx   <- nrow(X)
  lengthy <- length(y)
  if(nrowx != lengthy) stop("The number of rows of 'X' must agree with the length of y.\n")
  if(missing(learnind)) learnind <- 1:nrowx
  if(length(learnind) > nrowx) stop("The length of 'learnind' must be smaller than the number of observations.\n")
  y         <- as.factor(y)
  levels(y) <- 1:nlevels(y)
  if(nlevels(y) > 2){ 
    stop("The method is created for binary classification only. \n")
  } else {mode <- "binary"}
  y <- as.numeric(y)-1
  Ylearn <- y[learnind]
  Xlearn <- X[learnind,]
  Xtest  <- X[-learnind,,drop=FALSE]
  if(nrow(Xtest) == 0){
    Xtest <- Xlearn ; Ytest <- Ylearn} 
  else {Ytest <- y[-learnind]}

  pred   <- rlda.iter(Xlearn,Xtest,Ylearn,type,genesINpaths)
  new("cloutput",yhat=as.numeric(pred),y=Ytest,learnind=learnind,method="rldaCMA",mode=mode)
}


########################
# Function rlda.iter() #
########################

rlda.iter  <- function(Xlearn,Xtest,Ylearn,type,genesINpaths=NULL) {             
  genes  <- genesINpaths[colnames(Xlearn)]
  # The 'prior probabilities' are estimated through a shrinkage procedure                                                            
  # according to Hausser and Strimmer (2009).
  prior  <- freqs.shrink(table(Ylearn),verbose=FALSE)
  if (sum(prior)!=1) stop("The sum of prior probabilities must be 1! \n")
              
  mean0  <- colMeans(Xlearn[Ylearn==0,])
  mean1  <- colMeans(Xlearn[Ylearn==1,])
              
  sigma <-  shrink.estim.pool(Xlearn,Ylearn,build.target(Xlearn,Ylearn,genes,type=type))[[1]]             
  sigmainv <- pseudoinverse(sigma)

  discr     <- matrix(nrow=2,ncol=nrow(Xtest))
  for (j in 1:nrow(Xtest)) {
    discr[1,j]<- (Xtest[j,]%*%sigmainv)%*%mean0 - 0.5*(mean0%*%sigmainv)%*%mean0 + log(prior[1])
    discr[2,j]<- (Xtest[j,]%*%sigmainv)%*%mean1 - 0.5*(mean1%*%sigmainv)%*%mean1 + log(prior[2])
  }
  predclass <- apply(discr,MARGIN=2,FUN=which.max)-1
  return(predclass)
}